/*
 * segments.c
 *
 *  Created on: 06.07.2013
 *      Author: ralf
 */
#include <stdio.h>
#include "segments.h"

/**
 * Funktions-Prototypen
 */
void init_Display(void);
void do_Latch(uint8_t);
void send_Data(uint8_t, uint8_t);


// Externe Referenzen
extern void delayMillis(uint32_t);

/**
 * Konstanten
 */
const Pin bcdData[] = { BCD_DATA };
const Pin latches[] = { ALL_LATCHES };
const Pin pinLatch1 = PIN_LATCH_1;
const Pin pinLatch2 = PIN_LATCH_2;
const Pin pinLatch3 = PIN_LATCH_3;
const Pin pinLatch4 = PIN_LATCH_4;

/**
 * Globale Variablen
 */
uint8_t dig_1 = 0;    // Einer-Stelle
uint8_t dig_2 = 0;    // Zehner-Stelle
uint8_t dig_3 = 0;    // Hunderter-Stelle
uint8_t dig_4 = 0;    // Tausender-Stelle


/**
 * init_Display
 *
 * @param  none
 * @return none
 *
 * \brief  Initialisierung des Displays. Nach dem Start des Programms
 *         wechselt die Anzeige 5x zwischen '4321' und '0000'. Am
 *         Ende der Initialisierung wechselt die Anzeige auf '0000'.
 */
void init_Display(void)
{
	uint8_t i;

    // Konfiguration der verwendeten Portleitungen
    PIO_Configure(bcdData, DATA_SIZE);
    PIO_Configure(latches, LATCH_SIZE);

    for (i = 0; i < 5; i++)
    {
        send_Data(1, LATCH_1);
        send_Data(2, LATCH_2);
        send_Data(3, LATCH_3);
        send_Data(4, LATCH_4);

        delayMillis(500);

        send_Data(0, LATCH_1);
        send_Data(0, LATCH_2);
        send_Data(0, LATCH_3);
        send_Data(0, LATCH_4);

        delayMillis(500);
    }

    dig_1 = dig_2 = dig_3 = dig_4 = 0;
    send_Data(dig_1, LATCH_1);
    send_Data(dig_2, LATCH_2);
    send_Data(dig_3, LATCH_3);
    send_Data(dig_4, LATCH_4);
}


/**
 * do_Latch
 *
 * @param  uint8_t pinNummer fuer Latch-Impuls
 * @return none
 *
 * \brief  Vier 7-Segement-Displays erfordern 4x 74HCT4511. Jeder 74HCT4511
 *         hat einen eigenen Latch-Enable-Eingang, der fuer 2ms auf '0'
 *         gezogen und anschl. wieder auf '1' gesetzt wird.
 */
void do_Latch(uint8_t pinNumber)
{
	switch (pinNumber)
	{
	    case LATCH_1:
	    	delayMillis(1);
	    	PIO_Clear(&pinLatch1);
	    	delayMillis(2);
	    	PIO_Set(&pinLatch1);
	    	break;

	    case LATCH_2:
	    	delayMillis(1);
	    	PIO_Clear(&pinLatch2);
	    	delayMillis(2);
	    	PIO_Set(&pinLatch2);
	    	break;

	    case LATCH_3:
	    	delayMillis(1);
	    	PIO_Clear(&pinLatch3);
	    	delayMillis(2);
	    	PIO_Set(&pinLatch3);
	    	break;

	    case LATCH_4:
	    	delayMillis(1);
	    	PIO_Clear(&pinLatch4);
	    	delayMillis(2);
	    	PIO_Set(&pinLatch4);
	    	break;
	}
}


/**
 * send_Data
 *
 * @param  uint8_t data  --> BCD-Kode
 * @param  uint8_t latch --> Nummer des 74HCT4511, auf dem der BCD-Kode angezeigt werden soll
 * @return none
 *
 * \brief  Sendet den BCD-Kode vom Mikrocontroller zu dem durch 'latch' angegebenen 74HCT4511.
 *         Da die Daten nicht ueberschrieben werden, muessen sie beim Verlassen des Unterprogramms
 *         geloescht werden.
 */
void send_Data(uint8_t data, uint8_t latch)
{
	PIOA->PIO_CODR = 0x0F;
    PIOA->PIO_SODR = data;
    do_Latch(latch);
    PIOA->PIO_CODR = data;
}

