/**
 * hd44780.c
 *
 *  Created on: 29.10.2013
 *      Author: ralf
 *
 * \brief Initialisierung und Anwendung eines einfachen Text-LC-Displays
 *        auf Basis des Hitachi HD44780 LCD-Controllers oder kompatibler
 *        Typen, wie z.B. KS0066 oder KS0070.
 */
#include <stdint.h>

#include "board.h"
#include "chip.h"
#include "global.h"
#include "hd44780.h"


/**
 * Konstanten
 */
const Pin LCD_DATA_ARRAY[]    = { LCD_DATA };
const Pin LCD_CONTROL_ARRAY[] = { LCD_CONTROL };


/**
 * Globale Variablen
 */
// Speicherung des Blink-Modus pro Zeile
uint8_t blinkModeLine1;
uint8_t blinkModeLine2;
uint8_t blinkModeLine3;
uint8_t blinkModeLine4;
uint8_t tickerLine1Started = 0;
uint8_t tickerLine2Started = 0;
uint8_t tickerLine3Started = 0;
uint8_t tickerLine4Started = 0;
char    tickerLine1[17];
char    tickerLine2[17];
char    tickerLine3[17];
char    tickerLine4[17];


/**
 * static-Funktionen
 */
static void HD44780_Write(uint8_t, uint8_t);
static void HD44780_Write_Nibble(uint8_t, uint8_t);

/**
 * HD44780_Configure
 *
 * @param  none
 * @return none
 *
 * \brief Initialisierung des LC-Displays mit Einstellungen, die
 *        in der Datei HD44780.h definiert sind.
 */
void HD44780_Configure(void)
{
	uint8_t mode = 0;

    // Konfiguriere alle Portleitungen, die vom LC-Display genutzt werden
    PIO_Configure(LCD_DATA_ARRAY, PIO_LISTSIZE(LCD_DATA_ARRAY));
    PIO_Configure(LCD_CONTROL_ARRAY, PIO_LISTSIZE(LCD_CONTROL_ARRAY));

    // LCD-Portleitungen --> zuruecksetzen auf '0'
    PIOA->PIO_CODR = 0xFFFFFFFF;


#ifdef LCD_MODE_8_BIT
    // Warte mind. 30ms nach Power-on
	delayMillis(40);

	// 3x Initialisierungssequenz senden
	HD44780_Write(INSTR_WR, LCD_PRE_INIT_8_BIT);
	delayMillis(5);
	HD44780_Write(INSTR_WR, LCD_PRE_INIT_8_BIT);
	delayMillis(1);
	HD44780_Write(INSTR_WR, LCD_PRE_INIT_8_BIT);

	mode = LCD_SF | LCD_SF_DATA_8_BIT | LCD_SF_2_LINES | LCD_SF_FONT_SMALL;
	HD44780_Write(INSTR_WR, mode);

#else
    // Warte mind. 30ms nach Power-on
	delayMillis(40);

	// 3x Initialisierungssequenz senden
	HD44780_Write_Nibble(INSTR_WR, LCD_PRE_INIT_4_BIT);
	delayMillis(5);
	HD44780_Write_Nibble(INSTR_WR, LCD_PRE_INIT_4_BIT);
	delayMillis(1);
	HD44780_Write_Nibble(INSTR_WR, LCD_PRE_INIT_4_BIT);

	HD44780_Write_Nibble(INSTR_WR, LCD_RETURN_HOME);
	mode = LCD_SF | LCD_SF_DATA_4_BIT | LCD_SF_2_LINES | LCD_SF_FONT_SMALL;
	HD44780_Write(INSTR_WR, mode);

#endif // LCD_MODE_8_BIT

    mode = LCD_DM | LCD_DM_BLINK_OFF | LCD_DM_CURSOR_OFF | LCD_DM_ON;
    HD44780_Write(INSTR_WR, mode);

	mode = LCD_DISP_CLEAR;
	HD44780_Write(INSTR_WR, mode);

    delayMillis(2);
    mode = LCD_EMS | LCD_EMS_CURS_MOVE_INC;
    HD44780_Write(INSTR_WR, mode);
    mode = LCD_DM | LCD_DM_BLINK_OFF | LCD_DM_CURSOR_OFF | LCD_DM_ON;
    HD44780_Write(INSTR_WR, mode);
}

/**
 * HD44780_Write
 *
 * @param command
 * @param data
 * @return none
 *
 * \brief HD44780_Write sendet Steuerkommandos und Daten an den
 *        LCD-Controller. Die Unterscheidung zwischen Steuerdaten
 *        und Daten, die angezeigt werden sollen, wird durch den
 *        Parameter <b>command</b> bestimmt.
 */
static void HD44780_Write(uint8_t command, uint8_t data)
{
    if (command == INSTR_WR)
        PIO_Clear((Pin *) &LCD_CONTROL_ARRAY[PIN_RS]);
    else
        PIO_Set((Pin *) &LCD_CONTROL_ARRAY[PIN_RS]);

#ifdef LCD_MODE_8_BIT

    // Nach RS ~ 40ns warten, dann E high setzen. Wir waehlen 50s.
    delayMikros(50);
    PIO_Set((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);

    // E muss mind. 220ns high sein, dann Daten senden. Wir waehlen 10s.
    delayMikros(10);
    PIOA->PIO_SODR = data;

    // Daten muessen mind. 60ns anstehen, bevor E auf low geht.
    // Wir lassen die Daten 10s anstehen und setzen dann E auf low.
    delayMikros(10);
    PIO_Clear((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);

    // Daten muessen noch mind. weitere 20ns anliegen. Wir waehlen 1ms
    // und setzen dann den Datenbus zurueck
    delayMillis(1);
    PIOA->PIO_CODR = data;

#else   // LCD_MODE_4_BIT

    // Nach RS ~ 40ns warten, dann E high setzen. Wir waehlen 50s.
    delayMikros(50);
    PIO_Set((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);

    // E muss mind. 220ns high sein, dann Daten senden. Wir waehlen 10s.
    delayMillis(1);
    PIOA->PIO_SODR = data;  // 1. Nibble (4 Bit)

    // Daten muessen mind. 60ns anstehen, bevor E auf low geht.
    // Wir lassen die Daten 10s anstehen und setzen dann E auf low.
    delayMikros(10);
    PIO_Clear((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);

    // Daten (1. Nibble) zuruecksetzen
    PIOA->PIO_CODR = data;
    delayMillis(1);

    // Vorbereitung 2. Nibble
    data <<= 4;

    // Timing wie oben...
    PIO_Set((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);
    delayMikros(10);
    PIOA->PIO_SODR = data;

    delayMikros(10);
    PIO_Clear((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);

    PIOA->PIO_CODR = data;
    delayMillis(1);

#endif // LCD_MODE_8_BIT
}


/**
 * HD44780_Write_Nibble
 *
 * @param command
 * @param data
 * @return none
 *
 * \brief HD44780_Write_Nibble sendet 4-Bit-Steuerkommandos und
 *        Daten an den LCD-Controller. Die Unterscheidung zwischen
 *        Steuerdaten und Daten, die angezeigt werden sollen, wird
 *        durch den Parameter <b>command</b> bestimmt.
 */
static void HD44780_Write_Nibble(uint8_t command, uint8_t data)
{
    if (command == INSTR_WR)
        PIO_Clear((Pin *) &LCD_CONTROL_ARRAY[PIN_RS]);
    else
        PIO_Set((Pin *) &LCD_CONTROL_ARRAY[PIN_RS]);

    // Nach RS ~ 40ns warten, dann E auf high setzen
    delayMikros(50);
    PIO_Set((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);

    data <<= 4;
    delayMillis(1);
    PIOA->PIO_SODR = data;
    delayMillis(1);
    PIO_Clear((Pin *) &LCD_CONTROL_ARRAY[PIN_ENABLE]);
    delayMillis(1);

    PIOA->PIO_CODR = data;
}

/**
 * HD44780_PrintText
 *
 * @param orientation Ausrichtung des Textes auf dem Display
 * @param line        Zeilennummer eines mehrzeiligen Displays
 * @param *string     Der Text, der ausgegeben werden soll
 * @return none
 *
 * \brief
 */
void HD44780_PrintText(uint8_t orientation, uint8_t line, char *string)
{
	uint8_t count = 0;
	uint8_t pos;
	uint8_t xpos;
	uint8_t length;


    length = 0;
	while (*string++)
		length++;

	// Berechnung der Textstartposition
	switch(orientation)
    {
        case LCD_TEXT_LEFT :
        	xpos = 0;
        	break;

        case LCD_TEXT_CENTER :
        	xpos = (CHARS_PER_LINE - length) / 2;
        	break;

        case LCD_TEXT_RIGHT :
        	xpos = CHARS_PER_LINE - length;
        	break;
    }

	// Wenn angegebene Zeilennummer zu gross ist --> Setze
	// Ausgabezeile auf LCD_MAX_LINES
	if (line > LCD_MAXLINES)
		line = LCD_MAXLINES;

	switch (line)
	{
	    case LCD_LINE_1 :
	    	pos = LCD_OFFSET_LINE_1;
	    	break;

	    case LCD_LINE_2 :
	    	pos = LCD_OFFSET_LINE_2;
	    	break;

	    case LCD_LINE_3 :
	    	pos = LCD_OFFSET_LINE_3;
	    	break;

	    case LCD_LINE_4 :
	    	pos = LCD_OFFSET_LINE_4;
	    	break;
	}


    /* Setze Cursor-Position */
	HD44780_Write(INSTR_WR, pos + xpos);

    // String auf Startposition zuruecksetzen und dann jedes
    // Zeichen einzeln ausgeben
    string -= (length + 1);
    while (string[count] != 0) {
    	HD44780_Write(DATA_WR, string[count]);
        count++;
    }
}


/**
 * HD44780_BlinkText
 *
 * @param orientation   Textausrichtung
 * @param line          Zeilennummer der Zeile, die blinken soll
 * @param *string       Zeiger auf den Text, der angezeigt werden soll
 */
void HD44780_BlinkText(uint8_t  orientation, uint8_t  line, char *string)
{
	uint8_t pos;
	uint8_t xpos  = 0;
	uint8_t blinkMode;
	char    *emptyString = "                ";

	switch (line)
	{
	    case LCD_LINE_1 :
	    	pos = LCD_OFFSET_LINE_1;
	    	if (blinkModeLine1 == LCD_SINGLE_LINE_BLINK_OFF)
	    		blinkModeLine1 = LCD_SINGLE_LINE_BLINK_ON;
	    	else
	    		blinkModeLine1 = LCD_SINGLE_LINE_BLINK_OFF;

	    	blinkMode = blinkModeLine1;
	    	break;

	    case LCD_LINE_2 :
	    	pos = LCD_OFFSET_LINE_2;
	    	if (blinkModeLine2 == LCD_SINGLE_LINE_BLINK_OFF)
	    		blinkModeLine2 = LCD_SINGLE_LINE_BLINK_ON;
	    	else
	    		blinkModeLine2 = LCD_SINGLE_LINE_BLINK_OFF;

	    	blinkMode = blinkModeLine2;
            break;

	    case LCD_LINE_3 :
	    	pos = LCD_OFFSET_LINE_3;
	    	if (blinkModeLine3 == LCD_SINGLE_LINE_BLINK_OFF)
	    		blinkModeLine3 = LCD_SINGLE_LINE_BLINK_ON;
	    	else
	    		blinkModeLine3 = LCD_SINGLE_LINE_BLINK_OFF;

	    	blinkMode = blinkModeLine3;
	    	break;

	    case LCD_LINE_4 :
	    	pos = LCD_OFFSET_LINE_4;
	    	if (blinkModeLine4 == LCD_SINGLE_LINE_BLINK_OFF)
	    		blinkModeLine4 = LCD_SINGLE_LINE_BLINK_ON;
	    	else
	    		blinkModeLine4 = LCD_SINGLE_LINE_BLINK_OFF;

	    	blinkMode = blinkModeLine4;
	    	break;
	}

	if (blinkMode == LCD_SINGLE_LINE_BLINK_ON)
	{
	    /* Setze Cursor-Position */
		HD44780_Write(INSTR_WR, pos + xpos);
	    HD44780_PrintText(orientation, line, string);
	}
	else
	{
		HD44780_Write(INSTR_WR, pos + xpos);
		HD44780_PrintText(orientation, line, emptyString);
	}
}

/**
 * HD44780_TextTicker
 *
 * @param direction  Einstellung der Laufrichtung (links/rechts)
 * @param line       Zeilennummer
 * @param *string    Zeiger auf den Text, der angezeigt werden soll
 *
 * @return none
 *
 * \brief
 */
void HD44780_TextTicker(uint8_t direction, uint8_t line, char *string)
{
	uint8_t i = 0;
    uint8_t firstChar;
    uint8_t lastChar;
    uint8_t length = 0;

    if ((tickerLine1Started == 0) && (line = LCD_LINE_1))
    {
    	while ((char) *string)
    	{
    		tickerLine1[i++] = *string++;
    		length++;
    	}

        while (i < CHARS_PER_LINE)
        	tickerLine1[i++] = ' ';
        tickerLine1[16] = '\0';

    	tickerLine1Started = 1;
    }

    if ((tickerLine2Started == 0) && (line == LCD_LINE_2))
    {
    	while ((char) *string)
    	{
    		tickerLine2[i++] = *string++;
    		length++;
    	}

        while (i < CHARS_PER_LINE)
        	tickerLine2[i++] = ' ';
        tickerLine2[16] = '\0';

    	tickerLine2Started = 1;
    }

    if ((tickerLine3Started == 0) && (line == LCD_LINE_3))
    {
    	while ((char) *string)
    	{
    		tickerLine3[i++] = *string++;
    		length++;
    	}

        while (i < CHARS_PER_LINE)
        	tickerLine3[i++] = ' ';
        tickerLine3[16] = '\0';

    	tickerLine3Started = 1;
    }

    if ((tickerLine4Started == 0) && (line == LCD_LINE_4))
    {
    	while ((char) *string)
    	{
    		tickerLine4[i++] = *string++;
    		length++;
    	}

        while (i < CHARS_PER_LINE)
        	tickerLine4[i++] = ' ';
        tickerLine4[16] = '\0';

    	tickerLine4Started = 1;
    }

    if (direction == LCD_TICKER_LEFT)
    {
        i = 0;

        if (line == LCD_LINE_1)
        {
            firstChar = (uint8_t) tickerLine1[0];

            while (i < CHARS_PER_LINE)
            {
        	    tickerLine1[i] = tickerLine1[i+1];
        	    i++;
            }

            tickerLine1[CHARS_PER_LINE - 1] = firstChar;
            tickerLine1[CHARS_PER_LINE] = '\0';
        }

        if (line == LCD_LINE_2)
        {
            firstChar = (uint8_t) tickerLine2[0];

            while (i < CHARS_PER_LINE)
            {
        	    tickerLine2[i] = tickerLine2[i+1];
        	    i++;
            }

            tickerLine2[CHARS_PER_LINE - 1] = firstChar;
            tickerLine2[CHARS_PER_LINE] = '\0';
        }

        if (line == LCD_LINE_3)
        {
            firstChar = (uint8_t) tickerLine3[0];

            while (i < CHARS_PER_LINE)
            {
        	    tickerLine3[i] = tickerLine3[i+1];
        	    i++;
            }

            tickerLine3[CHARS_PER_LINE - 1] = firstChar;
            tickerLine3[CHARS_PER_LINE] = '\0';
        }

        if (line == LCD_LINE_4)
        {
            firstChar = (uint8_t) tickerLine4[0];

            while (i < CHARS_PER_LINE)
            {
        	    tickerLine4[i] = tickerLine4[i+1];
        	    i++;
            }

            tickerLine4[CHARS_PER_LINE - 1] = firstChar;
            tickerLine4[CHARS_PER_LINE] = '\0';
        }
    }
    else  // LCD_TICKER_RIGHT
    {
        i = CHARS_PER_LINE - 1;

        if (line == LCD_LINE_1)
        {
            lastChar = (uint8_t) tickerLine1[CHARS_PER_LINE - 1];

            while (i > 0)
   	        {
        	    tickerLine1[i] = tickerLine1[i - 1];
                i--;
   	        }

            tickerLine1[0] = lastChar;
            tickerLine1[CHARS_PER_LINE] = '\0';
        }

        if (line == LCD_LINE_2)
        {
            lastChar = (uint8_t) tickerLine2[CHARS_PER_LINE - 1];

            while (i > 0)
   	        {
        	    tickerLine2[i] = tickerLine2[i - 1];
                i--;
   	        }

            tickerLine2[0] = lastChar;
            tickerLine2[CHARS_PER_LINE] = '\0';
        }

        if (line == LCD_LINE_3)
        {
            lastChar = (uint8_t) tickerLine3[CHARS_PER_LINE - 1];

            while (i > 0)
   	        {
        	    tickerLine3[i] = tickerLine3[i - 1];
                i--;
   	        }

            tickerLine3[0] = lastChar;
            tickerLine3[CHARS_PER_LINE] = '\0';
        }

        if (line == LCD_LINE_4)
        {
            lastChar = (uint8_t) tickerLine4[CHARS_PER_LINE - 1];

            while (i > 0)
   	        {
        	    tickerLine4[i] = tickerLine4[i - 1];
                i--;
   	        }

            tickerLine4[0] = lastChar;
            tickerLine4[CHARS_PER_LINE] = '\0';
        }

    }

    if (line == LCD_LINE_1)
    	HD44780_PrintText(LCD_TEXT_LEFT, line, &tickerLine1[0]);

    if (line == LCD_LINE_2)
    	HD44780_PrintText(LCD_TEXT_LEFT, line, &tickerLine2[0]);

    if (line == LCD_LINE_3)
    	HD44780_PrintText(LCD_TEXT_LEFT, line, &tickerLine3[0]);

    if (line == LCD_LINE_4)
    	HD44780_PrintText(LCD_TEXT_LEFT, line, &tickerLine4[0]);
}


/**
 * void HD44780_ClearDisplay (void)
 *
 * \brief Loescht den Inhalt des LCD
 *
 * @param
 * @return  void
 */
void HD44780_ClearDisplay(void)
{
    uint8_t mode;

    mode = LCD_DISP_CLEAR;
    HD44780_Write(INSTR_WR, mode);
}


/**
 * void HD44780_ClearLine (uint8_t)
 *
 * \brief Loescht die angegebene Zeile auf dem LCD
 *
 * @param   uint8_t --> Zeilennummer
 * @return  void
 */
void HD44780_ClearLine(uint8_t lineNumber)
{
	uint8_t i;
    char    blankText[17] = "";

    for (i = 0; i < CHARS_PER_LINE; i++)
        blankText[i] = ' ';
    blankText[i] = '\0';

    HD44780_PrintText(LCD_TEXT_LEFT, lineNumber, blankText);
}


// EOF

