/** \file
 *
 *  This file contains all the specific code for the SDCard example.
 *
 */

/*----------------------------------------------------------------------------
 *        Headers
 *----------------------------------------------------------------------------*/

#include "board.h"
#include "sd_ll_spi1.h"
#include "sd_spi_mode.h"

#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <assert.h>

/*----------------------------------------------------------------------------
 *        Local definitions
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
 *        Local variables
 *----------------------------------------------------------------------------*/

/* this flag indicates that app has to periodically init the card */
static Boolean startInitializationFlag = FALSE;

/** time stamp */
uint32_t time_stamp = 0;

/*----------------------------------------------------------------------------
 *        Local functions
 *----------------------------------------------------------------------------*/

/**
 *  \brief Configure LEDs
 *
 *  Configures LEDs \#1 and \#2 (cleared by default).
 */
static void _ConfigureLeds(void)
{
	LED_Configure(0);
	LED_Configure(1);
}

/*----------------------------------------------------------------------------
 *        Exported functions
 *----------------------------------------------------------------------------*/
void SysTick_Handler(void)
{
	time_stamp++;
	/* simply to get 3s interval*/
	if (time_stamp % 3000 == 0)
	{
		/* start recognising the card*/
		startInitializationFlag = TRUE;
	}
}

/**
 * Application entry point.
 */
int main(void)
{
	pDiskCtrlBlk_t sdCardInfo;

	/* Disable watchdog */
	WDT_Disable( WDT);

	/* Output example information */
	printf("-- SDCARD Example %s --\n\r", SOFTPACK_VERSION);
	printf("-- %s\n\r", BOARD_NAME);
	printf("-- Compiled: %s %s --\n\r", __DATE__, __TIME__);

	printf("Configure LED PIOs.\n\r");
	_ConfigureLeds();

	/*1 ms timer*/
	SysTick_Config(BOARD_MCK / (1000));

	while (1)
	{
		while (!startInitializationFlag)
			;

		if (startInitializationFlag)
		{
		    startInitializationFlag = FALSE;
		    LED_Clear(0);
			LED_Clear(1);

			//check for CP
			if (!SdPresent())
			{
				printf("Please put SD card and press any key... \n\r");
				continue;
			}
			LED_Set(0);

			//check for WP
			if (SdWriteProtect())
			{
				printf("!!!Card is write protected.\n\r");
			}

			SdDiskInit();
			sdCardInfo = SdGetDiskCtrlBkl();

			if (sdCardInfo->DiskStatus == DiskCommandPass)
			{
				printf("SD BLOCK SIZE: %uB\n\r", sdCardInfo->BlockSize);
				printf("SD CARD SIZE: %uB\n\r",
						sdCardInfo->BlockSize * sdCardInfo->BlockNumb);
				LED_Set(1);
			}
			else
			    printf("Karte nicht benutzbar\r\n");
		}
	}

}

