/**
 * Datei: rtc_timer.c
 * Autor: Ralf Jesse
 * Datum: 24.09.2013
 *
 * \brief
 *
 */


/**
 * Include-Dateien
 */
#include <stdio.h>
#include <stdbool.h>

#include "board.h"
#include "chip.h"
#include "global.h"


/**
 * Configure_RTC
 *
 * \brief Standardkonfiguration des RTC.
 *
 * @param  none
 * @return none
 */
void Configure_RTC(void)
{
    RTC_SetHourMode(RTC, MODE_24H);

    NVIC_DisableIRQ(RTC_IRQn);
    NVIC_ClearPendingIRQ(RTC_IRQn);
    NVIC_SetPriority(RTC_IRQn, IRQ_PRIOR_PIO);
    NVIC_EnableIRQ(RTC_IRQn);

    // Interrupt wird ausgelst a) pro Sekunde, b) Weckzeit erreicht
    RTC_EnableIt(RTC, RTC_IER_SECEN | RTC_IER_ALREN);
}

/**
 * RTC_IrqHandler
 *
 * \brief Interrupt-Service-Routine zur Behandlung von Interrupts
 *        im Sekundentakt und des Weckers
 *
 * @param  none
 * @return none
 */
void RTC_IrqHandler(void)
{
	// Auslesen des RTC-Statusregisters
	uint32_t rtcStatus = RTC->RTC_SR;

	// Prfen, ob es sich bei dem RTC-Interrupt um ein Sekunden-Event handelt
	if ((rtcStatus & RTC_SR_SEC) == RTC_SR_SEC)
	{
		// Weitere RTC-Interrupts zunchst deaktivieren
		RTC_DisableIt(RTC, RTC_IDR_SECDIS);

		// Flag zur Verwendung in der main()-Funktion
		secondEvent = true;

        // Reset des Statusbits fr das Sekunden-Event
		// Anschl. reaktivieren des RTC-Interrupts
		RTC_ClearSCCR(RTC, RTC_SCCR_SECCLR);
		RTC_EnableIt(RTC, RTC_IER_SECEN);
	}

	// Auswertung des Alarms (Weckerfunktion)
	if ((rtcStatus & RTC_SR_ALARM) == RTC_SR_ALARM)
	{
	    RTC_DisableIt(RTC, RTC_IDR_ALRDIS);

	    // Flag zur Verwendung in der main()-Funktion
	    alarmEvent = true;

	    RTC_ClearSCCR(RTC, RTC_SCCR_ALRCLR);
	    RTC_EnableIt(RTC, RTC_IER_ALREN);
	}
}

/**
 * setClockTime
 *
 * \brief Einstellung der Uhrzeit im RTC
 *
 * @param  uint32_t clockMode
 * @param  uint8_t  clockHours
 * @param  uint8_t  clockMinutes
 * @param  uint8_t  clockSeconds
 *
 * @return none
 */
void setClockTime(uint32_t clockMode, uint8_t clockHours, uint8_t clockMinutes, uint8_t clockSeconds)
{
    RTC_SetHourMode(RTC, clockMode);
    RTC_SetTime(RTC, clockHours, clockMinutes, clockSeconds);

    // Uhrzeit ist gestellt: Programmstatus auf Normalbetrieb zurcksetzen
    currentState = STATE_NORMAL_RUN;
}


/**
 * setAlarmTime
 *
 * \brief Einstellung der Weckzeit im RTC
 *
 * @param  uint32_t clockMode
 * @param  uint8_t  alarmHours
 * @param  uint8_t  alarmMinutes
 * @param  uint8_t  alarmSeconds
 *
 * @return none
 */
void setAlarmTime(uint32_t clockMode, uint8_t alarmHours, uint8_t alarmMinutes, uint8_t alarmSeconds)
{
    RTC_SetHourMode(RTC, clockMode);
    RTC_SetTimeAlarm(RTC, &alarmHours, &alarmMinutes, &alarmSeconds);

    alarmCanceled = false;

    // Weckzeit ist gestellt: Programmstatus auf Normalbetrieb zurcksetzen
    currentState = STATE_NORMAL_RUN;
}


/**
 * setCalendarDate
 *
 * \brief Einstellung des Kalenders im RTC
 *
 *        ACHTUNG:
 *        Das Jahr lie sich auf unserem Entwicklungsboard nicht auf 19xx
 *        setzen: Beim Auslesen des Datums ergab sich immer das Jahr 20xx.
 *        Dieser Effekt trat immer auf: Mit der CMSIS-Funktion RTC_SetDate()
 *        und mit einer eigenen Version dieser Funktion, die nachfolgend
 *        erhalten geblieben ist.
 *
 * @param  uint32_t clockMode
 * @param  uint8_t  alarmHours
 * @param  uint8_t  alarmMinutes
 * @param  uint8_t  alarmSeconds
 *
 * @return none
 */
void setCalendarDate(uint16_t year, uint8_t month, uint8_t day, uint8_t dow)
{
    uint32_t newDate;
    uint8_t  scCentury;
    uint8_t  scYear;
    uint8_t  scMonth;
    uint8_t  scDay;
    uint8_t  scDow;

    scCentury = year / 100;
    scYear    = year % 100;
    scMonth   = month;
    scDay     = day;
    scDow     = dow;

    newDate   = (((scCentury / 10) << 4) | (scCentury % 10)) |
                (((scYear / 10) << 12) | ((scYear % 10) << 8)) |
                (((scMonth / 10) << 20) | ((scMonth % 10) << 16)) |
                (scDow << 21) |
                (((scDay / 10) << 28) | ((scDay % 10) << 24));

    // Der Einsatz der Funktion RTC_SetDate() ist hier ebenfalls
    // mglich!
    RTC->RTC_CR |= RTC_CR_UPDCAL ;
    while ((RTC->RTC_SR & RTC_SR_ACKUPD) != RTC_SR_ACKUPD) ;

    RTC->RTC_SCCR  = RTC_SCCR_ACKCLR;
    RTC->RTC_CALR  = newDate ;
    RTC->RTC_CR   &= (uint32_t)(~RTC_CR_UPDCAL) ;
    RTC->RTC_SCCR |= RTC_SCCR_SECCLR; /* clear SECENV in SCCR */

    // Das Datum wurde gestellt --> Rckkehr in Normalbetrieb!
	currentState = STATE_NORMAL_RUN;
}


/**
 * setAlarmTime
 *
 * \brief Einstellung des Alarmkalenders im RTC
 *
 * @param  uint32_t clockMode
 * @param  uint8_t  alarmHours
 * @param  uint8_t  alarmMinutes
 * @param  uint8_t  alarmSeconds
 *
 * @return none
 */
void setAlarmDate(void)
{
    // Die Implementierung dieser Funktion mag als kleine bung betrachtet werden.
}


// EOF
