/**
 * stringutil.c
 * ============
 *
 * \brief Enthaelt Hilfsfunktionen zur String-Konvertierung und zur Ausgabe
 *        der Ergebnisse auf dem Bildschirm und in eine Datei.
 *
 *        Provides helper functions for the handling of string and to print
 *        the results to the screen, or into a file.
 *
 * Datum/Date  : 26.11.2013
 * Autor/Author: Ralf Jesse
 */
#include <stdint.h>
#include <math.h>
#include <stdio.h>
#include <stdbool.h>
#include <string.h>

#include "global.h"
#include "stringutil.h"
#include "mathutil.h"

// Funktionsprototyp fuer statische Funktionen
static uint8_t findDecimalPoint(float);

/**
 * char * stringToLowerCase(char *string)
 *
 * \brief Umwandlung einer Zeichenkette in Kleinbuchstaben
 *
 * @param  string
 * @return string (umgewandelt)
 */
char *stringToLowerCase(char *string)
{
    int i;
    int len = strlen(string);
    for (i = 0; i < len; i++)
    {
        if (string[i] >= 'A' && string[i] <= 'Z')
        {
            string[i] += 32;
        }
    }
    return string;
}

/**
 * int stringToNumber(char *string)
 *
 * \brief Wandelt einen String in einen Integerwert. Das Ergebnis
 *        wird auf das naechstkleinere Vielfache von 10 abgerundet.
 *
 * @param  string
 * @return wert
 */
int stringToNumber(char *string)
{
    int value = 0;
    int i;
    int len = strlen(string);
    int currentChar;

    for (i = 0; i < len; i++)
    {
        currentChar = string[len - i - 1] - '0';

        if (currentChar >= 0 && currentChar < 10)
            value += currentChar * (ipow(10, i));    // ACHTUNG: ipow ist wie pow, aber fuer int-Werte!
        else
            return -1;
    }

    // Abrunden auf naechstkleineres Vielfaches von 10
    if ((value % 10) != 0)
    {
        value     = (value / 10) * 10;
    }

    return value;
}


/**
 * char* intToString (int16_t number)
 *
 * \brief Wandelt einen vorzeichenbehafteten 16-Bit-Integerwert in einen String um.
 *        Die Funktion ist fuer die Verwendung mit dem ADC/DACC vorgesehen. Da diese
 *        eine maximale Aufloesung von 12 Bit haben, reicht fuer die Umwandlung des
 *        Integerwertes ein 16-Bit-Wert. Die maximale Stellenzahl betraegt vier Stellen
 *        (-4095 --> 0 --> +4095).
 *
 * @param   number
 * @return  char*
 */
char *intToString(int16_t number)
{
    uint16_t divisor = 1000;
    uint16_t tempVal;
    uint8_t  i = 0;
    uint8_t  n;

    // Vorzeichen ermitteln
    if (number < 0)
    {
        retVal[i++] = '-';
        number *= (-1);        // Absolutbetrag bilden
    }
    else
    	retVal[i++] = '+';

    // Das Vorzeichen wurde bereits ermittelt --> daher: Start mit Index = 1
    for (n = 0; n < 4; n++)
    {
        tempVal = number / divisor;
        retVal[i++] = tempVal + '0';
        number -= tempVal * divisor;
        divisor /= 10;
    }
    retVal[i] = '\0';

    return retVal;
}


/**
 * void intToCharArray (int16_t, char[])
 *
 * \brief Wandelt einen vorzeichenbehafteten 16-Bit-Integerwert in ein char[] um.
 *        Die Funktion ist fuer die Verwendung mit dem ADC/DACC vorgesehen. Da diese
 *        eine maximale Aufloesung von 12 Bit haben, reicht fuer die Umwandlung des
 *        Integerwertes ein 16-Bit-Wert. Die maximale Stellenzahl betraegt vier Stellen
 *        (-4095 --> 0 --> +4095).
 *
 * @param   int16_t  Wert, der zu konvertieren ist
 * @param   char[]   Array zur Aufnahme des konvertierten Wertes
 * @return  void
 */
uint8_t intToCharArray(int16_t number, char result[])
{
    uint8_t  offset  = 0;
    uint16_t divisor = 1000;
    uint16_t tempVal;
    uint8_t  n;

    // Vorzeichen und Absolutbetrag von 'number' ermitteln
    if (number < 0)
    {
    	retVal[offset++] = '-';
        number  *= (-1);
    }
    else
    	retVal[offset++] = '+';

    for(n = 0; n < 4; n++)
    {
        tempVal = number / divisor;
        retVal[offset++] = tempVal + '0';
        number -= tempVal * divisor;
        divisor /= 10;
    }
    retVal[offset] = '\0';

    return offset;
}


/**
 * void floatToCharArray (float, char[])
 *
 * \brief Wandelt eine vorzeichenbehaftete Fliesspunktzahl in ein char[] um.
 *        Es wird nur die erste Nachkommastelle beruecksichtigt (ohne Rundung).
 *
 * @param   float  : Die umzuwandelnde Fliesskommazahl
 * @param   char[] : Array, das die Fliesskommazahl als 'char' aufnimmt
 * @return  void
 */
uint8_t floatToCharArray(float number, char result[])
{
	uint8_t  offset  = 0;
	uint8_t  dpPos   = 0;    // Position des Dezimalpunktes
	uint8_t  i, n;
	uint16_t divisor = 1;
	uint16_t tempVal;
	uint8_t  oneDigit;

    // Vorzeichen und Absolutbetrag von 'number' ermitteln
	if (number < 0)
    {
        retVal[offset++] = '-';
        number *= (-1);
    }
    else
        retVal[offset++] = '+';

    // Position des Dezimalpuntes ermitteln und hieraus den Divisor des
	// ganzzahligen Teils von 'number' bestimmen. Der gefundene Wert fuer
	// dpPos muss um '1' verkleinert werden, weil der letzte Nutzwert VOR
	// dem Dezimalpunkt steht!
	dpPos = findDecimalPoint(number);
    for (i = 0; i < (dpPos - 1); i++)
    {
        divisor *= 10;
    }

    for (n = 0; n < dpPos; n++)
    {
        tempVal = number / divisor;
        retVal[offset++] = tempVal + '0';
        number -= tempVal * divisor;
        divisor /= 10;
    }
    retVal[offset++] = '.';    // Dezimalpunkt einsetzen

    // 1. Nachkommastelle vor den Dezimalpunkt "ziehen"
    number *= 10;
    oneDigit = (uint8_t) number;

    // Nachkommastelle(n)
    retVal[offset++] = oneDigit + '0';
    retVal[offset]   = '\0';

    return offset;
}


/**
 * uint8_t findDecimalPoint (float)
 *
 * \brief  Findet die Position des Dezimalpunktes in einer Fliesskommazahl.
 *
 * @param   float
 * @return  uint8_t
 */
static uint8_t findDecimalPoint(float number)
{
    uint8_t  dpPos = 0;
    uint8_t  msb;

    do
    {
        number /= 10;
        msb = (uint8_t) number;
        dpPos++;
    } while (msb != 0);

    return dpPos;
}

// EOF (util.c)
