#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <stdint.h>     // Typdefinitionen, z.B. uint8_t oder uint16_t
#include <termios.h>
#include <string.h>     // Benoetigt fuer memcpy


#include "max7219_test.h"

int main (int argc, char **argv, char **envp)
{
    uint8_t  count_run;
    uint8_t  pos = 0;
    uint16_t counter;

    fd_gpio = open ("/dev/max7219", O_WRONLY);
    if (fd_gpio < 0)
    {
        printf ("Kann \"/dev/max7219\" nicht oeffnen.\n");
        return -1;
    }

    // Initialisierung des 7219
    max7219_init(fd_gpio,                   // Handle des Geraets
                 OPMODE_NORMAL,             // Betriebsart
                 MAX7219_DECODE_OFF,        // Keine Dekodierung
                 MAX7219_MAX_INTENSITY,     // Helligkeit einstellen
                 MAX7219_SCAN_DIG_0_7,      // 7 Spalten werden gescannt
                 3);                        // Anzahl der MAX7219

    max7219_set_intensity (fd_gpio, 0, MAX7219_MAX_INTENSITY);

#if 0
    for (count_run = 0; count_run < 50; count_run++)
    {
        max7219_exec_matrix_test(fd_gpio, pos, ON);
        usleep (250000);
        max7219_exec_matrix_test(fd_gpio, 0, OFF); // Alle Matrizen ausschalten
        usleep (250000);
        pos += 1;
        if (pos > 3)
        {
            pos = 0;
        }
    }
#endif // 0

    for (count_run = 0; count_run < 16; count_run++)
    {
        max7219_show_bitmap (fd_gpio, MAX7219_DIG2, count_run);
        for (counter = MAX7219_DIG3; counter <= MAX7219_DIG7; counter++)
        {
            max7219_show_bitmap (fd_gpio, counter, 0);
        }
        usleep (100000);
    }

    printf ("Fertig - Zum Beenden des Programms eine Taste druecken.\n");

#if 1
    while (1)
    {
        if (kbhit ())
        {
            max7219_exec_matrix_test(fd_gpio, 0, OFF);
            max7219_set_op_mode (fd_gpio, 0, OPMODE_SHUTDOWN);
            break;
        }
    }
#endif // 0

    close (fd_gpio);

    return 0;
}

/**
 * max7219_init
 * ============
 * @brief <B>max7219_init</b> initialisiert die Betriebsart der verwendeten
 * MAX7219-Bausteine. <B>max7219_init</b> muss aufgerufen werden, bevor eine der
 * anderen Funktionen verwendet werden kann.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  sdm   Shutdown-Modus.
 * @param[in]  dcm   Decode mode. Auswahl der Dekodierung
 * @param[in]  its   Intensity. Helligkeit der Anzeige.
 * @param[in]  scl   Scan limit. Anzahl der Spalten/Ziffern, die genutzt werden.
 * @param[in]  num   Anzahl der anzusteuerenden MAX7219-Bausteine.
 * @param[out] -
 */
void max7219_init( int     handle,
                   uint8_t sdm,
                   uint8_t dcm,
                   uint8_t its,
                   uint8_t scl,
                   uint8_t num
                 )
{
    uint16_t error;

    max7219_set_num (handle, num);

    max7219_set_op_mode (handle, 0, OPMODE_NORMAL);
    max7219_set_decode_mode (handle, 0, MAX7219_DECODE_OFF);
    max7219_set_intensity (handle, 0, MAX7219_MAX_INTENSITY);
    max7219_set_scan_limit (handle, 0, MAX7219_SCAN_DIG_0_7);

    max7219_init_done = 1;
}

/**
 * max7219_set_num
 * ===============
 * @brief <B>max7219_set_num</B> informiert den Treiber über die Anzahl der
 * MAX7219-Bausteine, die der Treiber ansteuern soll.
 *
 * @param[in]  dh   Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  sdm  Betriebsarten: OPMODE_SHUTDOWN und OPMODE_NORMAL.
 * @param[out] tbd
 */
int max7219_set_num (int handle, uint8_t num)
{
    uint8_t index = 0;

    data[index++] = MAX7219_SETNUM;
    data[index++] = num;
    write (handle, data, index);

    return 0;
}

/**
 * max7219_set_op_mode
 * ===================
 * @brief Mit <B>max7219_set_op_mode</b> erfolgt die Einstellung der
 * Betriebsart. Zulaessige Argumente sind <B>OPMODE_SHUTDOWN</B> und
 * <B>OPMODE_NORMAL</B>.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  pos   Position der Matrix/Ziffer, deren Betriebsart eingestellt
 *                   werden soll. Die Eingabe von '0' wirkt sich auf alle
 *                   Matrizen/Ziffern gleich aus.
 * @param[in]  sdm   Betriebsarten: OPMODE_SHUTDOWN und OPMODE_NORMAL.
 * @param[out] tbd
 */
int max7219_set_op_mode (int handle, uint8_t pos, uint8_t sdmode)
{
    uint8_t index = 0;

    memset (data, 0, sizeof (data));
    if ((sdmode != OPMODE_NORMAL) && (sdmode != OPMODE_SHUTDOWN))
    {
        return -ERR_MAX7219_INVALID_ARG;
    }

    data[index++] = MAX7219_OPMODE;
    data[index++] = pos;
    data[index++] = sdmode;
    write (handle, data, index);

    return 0;
}

/**
 * max7219_exec_matrix_test
 * ========================
 * @brief <B>max7219_exec_matrix_test</B> kann eine ueber die Angabe einer
 * Position genau spezifizierte Matrix oder Ziffer testen. Gezaehlt wird von
 * links nach rechts, wobei die erste Ziffer oder Matrix die Position '1' hat.
 * Wird als Position der Wert '0' angegeben, so werden alle Matrizen oder
 * Ziffern getestet.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  pos   Position der zu testenden Matrix oder Ziffer.
 * @param[out] tbd
 */
int max7219_exec_matrix_test (int handle, uint8_t pos, uint8_t mode)
{
    uint8_t index = 0;

    memset (data, 0, sizeof (data));
    data[index++] = MAX7219_DISPTEST;
    data[index++] = pos;
    data[index++] = mode;
    write (handle, data, index);

    return 0;
}

/**
 * max7219_set_intensity
 * =====================
 * @brief <B>max7219_set_intensity</B> steuert die Helligkeit einer Matrix oder
 * Ziffer. Ueber die Positionsangabe <B>pos</B> kann <B>max7219_set_intensity</B> die
 * Helligkeit einer genau spezifizierten Matrix oder Ziffer einstellen. Ist <B>pos</B>
 * = 0, so ist die Helligkeit aller Matrizen/Ziffern gleich.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  pos   Position der Matrix/Ziffer, deren Helligkeit eingestellt
 *                   werden soll. Die Eingabe von '0' stellt die Helligkeit
 *                   aller Matrizen/Ziffern gleich ein.
 * @param[in]  its   Helligkeitswert.
 * @param[out] tbd
 */
int  max7219_set_intensity (int handle, uint8_t pos, uint8_t intensity)
{
    uint8_t index = 0;

    if ((intensity < MAX7219_MIN_INTENSITY) ||
        (intensity > MAX7219_MAX_INTENSITY))
    {
        intensity = MAX7219_MED_INTENSITY;
    }

    memset (data, 0, sizeof (data));
    data[index++] = MAX7219_INTENSITY;
    data[index++] = pos;
    data[index++] = intensity;
    write (handle, data, index);

    return 0;
}

/**
 * max7219_set_decode_mode
 * =======================
 * @brief Mit <B>max7219_set_decode_mode</B> kann ueber die Positionsangabe <B>pos</B>
 * der Dekodiermodus für jede einzelne Matrix/Ziffer eingestellt werden. Hat <B>pos</B>
 * den Wert '0', so wird der eingestellte Dekodiermodus fuer die gesamte Anzeige
 * identisch eingestellt. Die Einstellung des Dekodiermodus ist nur bei
 * 7-Segment-Anzeigen sinnvoll. Bei Matrizen sollte hier MAX7219_DECODE_OFF
 * gewählt werden.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  pos   Position der Matrix/Ziffer, fuer die der Dekodiermodus
 *                   eingestellt werden soll. Die Eingabe von '0' stellt den
 *                   Dekodiermodus fuer alle Matrizen/Ziffern gleich ein.
 * @param[in]  dm    Dekodiermodus.
 * @param[out] tbd
 */
int  max7219_set_decode_mode (int handle, uint8_t pos, uint8_t mode)
{
    uint8_t index = 0;

    memset (data, 0, sizeof (data));
    data[index++] = MAX7219_DECODEMODE;
    data[index++] = pos;
    data[index++] = mode;
    write (handle, data, index);

    return 0;
}

/**
 * max7219_set_scan_limit
 * ======================
 * @brief <B>max7219_set_scan_limit</B> stellt die Anzahl der zu scannenden
 * Spalten oder Ziffern ein. Dies ermoeglicht die Verwendung unterschiedlicher
 * Anzeigen oder Matrizen.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  pos   Position der Ziffer oder Matrix.
 * @param[in]  cols  Anzahl der verwendeten Spalten einer Matrix/Ziffern einer
 *                   7-Segment-Anzeige.
 * @param[out] tbd
 */
int  max7219_set_scan_limit (int handle, uint8_t pos, uint8_t cols)
{
    uint8_t index = 0;

    memset (data, 0, sizeof (data));
    data[index++] = MAX7219_SCANLIMIT;
    data[index++] = pos;
    data[index++] = cols;
    write (handle, data, index);

    return 0;
}

/**
 * max7219_show_bitmap
 * ===================
 * @brief <B>max7219_show_bitmap</B>
 */
int  max7219_show_bitmap (int handle, uint8_t pos, uint8_t bitmap)
{
    uint8_t index = 0;

    memset (data, 0, sizeof (data));
    data[index++] = MAX7219_SHOW_BITMAP;
    data[index++] = pos;
    data[index++] = bitmap;
    write (handle, data, index);

    return 0;
}

/**
 * max7219_scroll_bitmap
 * =====================
 * @brief <B>max7219_scroll_bitmap</B> verschiebt das Bitmap der durch den
 * Parameter <B>pos</B> angegebenen LED-Matrix in die angegebene Richtung. Dies
 * ermoeglicht bei mehreren miteinander verketten LED-Matrizen unterschiedliche
 * Richtungen, in die das jeweilige Bitmap gerollt wird. Die Verwendung von
 * <B>max7219_scroll_bitmap</B> ist nur fuer LED-Matrizen sinnvoll.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  pos   Position der Ziffer oder Matrix.
 * @param[in]  dir   Richtung, in der das Bitmap gerollt werden soll. Zulaessige
 *                   Werte sind ROLL_LEFT, ROLL_RIGHT, ROLL_UP und ROLL_DOWN.
 * @param[in]  bm    Pointer auf das Bitmap fuer die gewaehlte Matrix.
 * @param[out] tbd
 */
int  max7219_scroll_bitmap ( int handle,
                             uint8_t pos,
                             uint8_t dir,
                             uint8_t *bitmap)
{
    return 0;
}

/**
 * max7219_spread_bitmap
 * =====================
 * @brief <B>max7219_spread_bitmap</B> kann ein beliebiges Bitmap in seiner
 * Hoehe und Breite wachsen lassen. Die Verwendung ist nur bei LED-Matrizen
 * sinnvoll.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  align Ausrichtung des Bitmaps (LEFT, CENTER, RIGHT).
 * @param[in]  wmin  Minimale Breite (width) des Bitmaps.
 * @param[in]  wmax  Maximale Breite (width) des Bitmaps.
 * @param[in]  hmin  Minimale Hoehe (height) des Bitmaps.
 * @param[in]  hmax  Maximale Hoehe (height) des BItmaps.
 * @param[out] tbd
 */
int  max7219_spread_bitmap ( int handle,
                             uint8_t align,
                             uint8_t wmin,
                             uint8_t wmax,
                             uint8_t hmin,
                             uint8_t hmax
                           )
{
    return 0;
}

/**
 * max7219_shrink_bitmap
 * =====================
 * @brief <B>max7219_shrink_bitmap</B> kann ein beliebiges Bitmap in seiner
 * Hoehe und Breite schrumpfen lassen. Die Verwendung ist nur bei LED-Matritzen
 * sinnvoll.
 *
 * @param[in]  dh    Device handle. Bezeichner für das angesprochene Gerät.
 * @param[in]  align Ausrichtung des Bitmaps (LEFT, CENTER, RIGHT).
 * @param[in]  wmin  Minimale Breite (width) des Bitmaps.
 * @param[in]  wmax  Maximale Breite (width) des Bitmaps.
 * @param[in]  hmin  Minimale Hoehe (height) des Bitmaps.
 * @param[in]  hmax  Maximale Hoehe (height) des BItmaps.
 * @param[out] tbd
 */
int  max7219_shrink_bitmap ( int handle,
                             uint8_t align,
                             uint8_t wmin,
                             uint8_t wmax,
                             uint8_t hmin,
                             uint8_t hmax
                           )
{
    return 0;
}


//-----  H I L F S F U K T I O N E N  -----

uint8_t getch (void)
{
    static int ch = -1, fd = 0;
    struct termios current, old;

    fd = fileno (stdin);
    tcgetattr (fd, &old);
    current = old;
    current.c_lflag &= ~ (ICANON | ECHO);
    tcsetattr (fd, TCSANOW, &current);
    ch = getchar ();
    tcsetattr (fd, TCSANOW, &old);
    return ch;
}

uint8_t kbhit(void)
{
    struct termios term, oterm;
    int fd = 0;
    int c = 0;

    tcgetattr(fd, &oterm);
    memcpy(&term, &oterm, sizeof(term));
    term.c_lflag = term.c_lflag & (!ICANON);
    term.c_cc[VMIN] = 0;
    term.c_cc[VTIME] = 1;
    tcsetattr(fd, TCSANOW, &term);
    c = getchar();
    tcsetattr(fd, TCSANOW, &oterm);
    if (c != -1)
        ungetc(c, stdin);
    return ((c != -1) ? 1 : 0);
}
