#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
#include <linux/time.h>

#include "max7219.h"


//#define DEBUG

//
//================= Komfortfunktionen des Treibers ===================
//

/**
 * max7219_set_num_devices
 * =======================
 * @brief <B>max7219_set_num_devices</B> legt fest, wieviele MAX7219 der Treiber
 * insgesamt in einer Anwendung verwenden soll.
 *
 * @param[in]   num      Anzahl der verwendeten MAX7219-Chips.
 */
static void max7219_set_num_devices (uint8_t num)
{
    if (num > 0)
    {
        max7219_count = num;
    }
}

/**
 * max7219_set_op_mode
 * ===================
 * @brief <B>max7219_set_op_mode</B> stellt die Betriebsart der verwendeten
 * MAX7219 ein. Die Betriebsart kann wahlweise <OPMODE_SHUTDOWN> oder
 * <OPMODE_NORMAL> sein.
 *
 * @param[in]   mode    Gewuenschte Betriebsart.
 * @param[out]  -
 */
static void max7219_set_op_mode (uint8_t pos, uint8_t mode)
{
    uint16_t value;

#ifdef DEBUG
    printk (KERN_ALERT "\nmax7219_set_op_mode(%d, %d)\n", pos, mode);
#endif // DEBUG

    value = (MAX7219_OPMODE << 8) | mode;
    gpio_set_value (GPIO24_LOAD, LOW);  // LOAD/CS = LOW setzen.
    max7219_make_data (pos, value);     // Bitdaten erzeugen
    gpio_set_value (GPIO24_LOAD, HIGH); // LOAD/CS = HIGH setzen.
    gpio_set_value (GPIO24_LOAD, LOW);  // Zum Schluss: LOAD/CS = LOW setzen.
}

/**
 * max7219_set_decode_mode
 * =======================
 * @brief <B>max7219_set_decode_mode</B> stellt den Dekodiermodus ein.
 *
 * @param[in]   mode  Dekodermodus
 */
static void max7219_set_decode_mode (uint8_t pos, uint8_t mode)
{
    uint16_t value;

#ifdef DEBUG
    printk (KERN_ALERT "\nmax7219_set_decode_mode(%d, %d)\n", pos, mode);
#endif // DEBUG

    value = (MAX7219_DECODEMODE << 8) | mode;
    gpio_set_value (GPIO24_LOAD, LOW);  // LOAD/CS = LOW setzen.
    max7219_make_data (pos, value);     // Bitdaten erzeugen
    gpio_set_value (GPIO24_LOAD, HIGH); // LOAD/CS = HIGH setzen.
    gpio_set_value (GPIO24_LOAD, LOW);  // Zum Schluss: LOAD/CS = LOW setzen.
}

/**
 * max7219_set_intensity
 * =====================
 * @brief <B>max7219_set_intensity</B> stellt die Helligkeit der verwendeten
 * Maxtrizen/7-Segment-Anzeigen ein. Zulaessige Werte liegen zwischen
 * <B>MAX7219_MIN_INTENSITY</B> und <B>MAX7219_MAX_INTENSITY</B>.
 *
 * @param[in]   its     Gewuenschte Helligkeit der Anzeigen
 */
static void max7219_set_intensity (uint8_t pos, uint8_t its)
{
    uint16_t value;

    if ((its < MAX7219_MIN_INTENSITY) || (its > MAX7219_MAX_INTENSITY))
    {
        its = MAX7219_MED_INTENSITY;
    }

#ifdef DEBUG
    printk (KERN_ALERT "\nmax7219_set_intensity(%d, %d)\n", pos, its);
#endif // DEBUG

    value = (MAX7219_INTENSITY << 8) | its;
    gpio_set_value (GPIO24_LOAD, LOW);  // LOAD/CS = LOW setzen.
    max7219_make_data (pos, value);     // Bitdaten erzeugen
    gpio_set_value (GPIO24_LOAD, HIGH); // LOAD/CS = HIGH setzen.
    gpio_set_value (GPIO24_LOAD, LOW);  // Zum Schluss: LOAD/CS = LOW setzen.
}

/**
 * max7219_set_scan_limit
 * ======================
 * @brief <B>max7219_set_scan_limit</B> stellt ein, wieviele Spalten jeder
 * MAX7219-Chip scannen soll.
 *
 * @param[in]   limit   Gewuenschte Betriebsart
 */
static void max7219_set_scan_limit (uint8_t pos, uint8_t limit)
{
    uint16_t value;

#ifdef DEBUG
    printk (KERN_ALERT "\nmax7219_set_scan_limit(%d, %d)\n", pos, limit);
#endif // DEBUG

    value = (MAX7219_SCANLIMIT << 8) | limit;
    gpio_set_value (GPIO24_LOAD, LOW);  // LOAD/CS = LOW setzen.
    max7219_make_data (pos, value);     // Bitdaten erzeugen
    gpio_set_value (GPIO24_LOAD, HIGH); // LOAD/CS = HIGH setzen.
    gpio_set_value (GPIO24_LOAD, LOW);  // Zum Schluss: LOAD/CS = LOW setzen.
}

/**
 * max7219_exec_disp_test
 * ======================
 * @brief <B>max7219_exec_disp_test</B> testet die angegebene Matrix/Stelle
 * einer 7-Segment-Anzeige. Wird hier der Wert 0 angegeben, so werden alle
 * Matrizen/Stellen getestet. Ist der Wert groesser als 0, so bezeichnet der
 * Wert die angegebene Matrix/Stelle.
 *
 * @param[in]   num     Gewuenschte Betriebsart
 */
static void max7219_exec_disp_test (uint8_t pos, uint8_t num)
{
    uint16_t value;

#ifdef DEBUG
    printk (KERN_ALERT "\nmax7219_exec_disp_test(%d, %d)\n", pos, num);
#endif // DEBUG

    value = (MAX7219_DISPTEST << 8) | num;
    gpio_set_value (GPIO24_LOAD, LOW);  // LOAD/CS = LOW setzen.
    max7219_make_data (pos, value);     // Bitdaten erzeugen
    gpio_set_value (GPIO24_LOAD, HIGH); // LOAD/CS = HIGH setzen.
    gpio_set_value (GPIO24_LOAD, LOW);  // Zum Schluss: LOAD/CS = LOW setzen.
}

/**
 * max7219_show_bitmap
 * ===================
 */
static void max7219_show_bitmap (uint8_t pos, uint8_t bitmap)
{
    uint16_t value;
    uint8_t  dig;

    value = (MAX7219_DIG0 << 8) | bitmap;
    gpio_set_value (GPIO24_LOAD, LOW);
    printk (KERN_ALERT "value = %d\n", value & 0x00FF);
    max7219_make_data (pos, value);
    gpio_set_value (GPIO24_LOAD, HIGH);
    gpio_set_value (GPIO24_LOAD, LOW);

#if 0
    value = (pos << 8) | bitmap;
    gpio_set_value (GPIO24_LOAD, LOW);  // LOAD/CS = LOW setzen.
    max7219_make_data (pos, bitmap);     // Bitdaten erzeugen
    gpio_set_value (GPIO24_LOAD, HIGH); // LOAD/CS = HIGH setzen.
    gpio_set_value (GPIO24_LOAD, LOW);  // Zum Schluss: LOAD/CS = LOW setzen.
#endif // 0
}

//
//================= Kernfunktionen des Treibers =================
//

/**
 * max7219_open
 * ============
 * <B>max7219_open</B> wird aus dem Userspace-Programm aufgerufen, wenn dort die
 * <B>open()</B>-Funktion verwendet wird.
 */
static int max7219_open (struct inode *device_file, struct file *instance)
{
    int err;
    int gpio_num;

    for (gpio_num = GPIO_OFFSET;
         gpio_num < GPIO_OFFSET + GPIOS_USED;
         gpio_num++)
    {
        // Anforderung der benoetigten GPIOs.
        err = gpio_request(gpio_num, "GPIO");
        if (err)
        {
            pr_err("%s: gpio_request() failed.\n", DEVICE_NAME);
            return -1;
        }

        // Alle GPIOs werden als Ausgang genutzt.
        err = gpio_direction_output(gpio_num, 0);
        if (err)
        {
            pr_err ("%s: gpio_direction_output() failed.\n", DEVICE_NAME);
            return -1;
        }
    }

    // Zu Beginn werden die Startpegel aller verwendeten Ports gesetzt.
    gpio_set_value(GPIO23_DIN, LOW);
    gpio_set_value(GPIO24_LOAD, LOW);
    gpio_set_value(GPIO25_CLK, LOW);

    dev_info (max7219_dev, "max7219_open called\n");

    // Alle Aufgaben erfolgreich abgeschlossen.
    return 0;
}

/**
 * max7219_close
 * =============
 * <B>max7219_close</B> wird aus dem Userspace-Programm aufgerufen, wenn das
 * Userspace-Programm beendet wird.
 */
static int max7219_close (struct inode *device_file, struct file *instance)
{
    int gpio_num;

#ifdef DEBUG
    printk (KERN_ALERT "\n\n---------------\n");
    printk (KERN_ALERT "Finally:\n");
    printk (KERN_ALERT "max7219_set_op_mode called from max7219_close\n");
    max7219_set_op_mode (0, OPMODE_SHUTDOWN);
#endif // DEBUG

    gpio_set_value(GPIO23_DIN, LOW);
    gpio_set_value(GPIO24_LOAD, LOW);
    gpio_set_value(GPIO25_CLK, LOW);

    for (gpio_num = GPIO_OFFSET;
         gpio_num < GPIO_OFFSET + GPIOS_USED;
         gpio_num++)
    {
        gpio_free(gpio_num);
    }
    dev_info (max7219_dev, "max7219_close called\n");

    return 0;
}

/**
 * max7219_write
 * =============
 * <B>max7219_write</B> wird aus dem Userspace aufgerufen, z.B. bei
 * <B>echo "text" > /dev/treiber</B>.
 */
static ssize_t max7219_write ( struct file *instance,   // Handle des Geraetes.
                               const char __user *user, // Daten aus Userspace.
                               size_t count,            // Daten: Anz. in Byte.
                               loff_t *offset           // Hier nicht benoetigt.
                             )
{
    unsigned long not_copied, to_copy;
    uint8_t       command;          // Funktion des Treibers.

    // Vor jeder Verwendung von data_buf wird dieser Speicher geleert. Danach
    // wird der Buffer mit den aktuellen Daten aus dem Userspace aufgefuellt.
    memset (buf, 0, sizeof (buf));

    to_copy = min (count, sizeof (buf));
    not_copied = copy_from_user (buf, user, to_copy);

    // Prueft, ob das High-Nibble des ersten Bytes >= 0x10 ist --> Wenn ja,
    // handelt es sich um eine Sonderfunktion des Treibers.
    if (buf[0] >= MAX7219_SET_NUM)
    {
        command = buf[0];
        switch (command)
        {
            case MAX7219_SET_NUM:
                max7219_set_num_devices ((uint8_t) buf [1]);
                break;
            case MAX7219_SHOW_BITMAP:
                max7219_show_bitmap (buf[1], buf[2]);
                break;
        }
    }
    else if ((buf[0] >= MAX7219_NOOP) && (buf[0] < MAX7219_SET_NUM))
    {
        command = buf[0] & 0x0F;
        switch (command)
        {
            case MAX7219_NOOP:
                break;
            case MAX7219_DIG0:
                break;
            case MAX7219_DIG1:
                break;
            case MAX7219_DIG2:
                break;
            case MAX7219_DIG3:
                break;
            case MAX7219_DIG4:
                break;
            case MAX7219_DIG5:
                break;
            case MAX7219_DIG6:
                break;
            case MAX7219_DIG7:
                break;
            case MAX7219_DECODEMODE:
                max7219_set_decode_mode (buf[1], buf[2]);
                break;
            case MAX7219_INTENSITY:
                max7219_set_intensity (buf[1], buf[2]);
                break;
            case MAX7219_SCANLIMIT:
                max7219_set_scan_limit (buf[1], buf[2]);
                break;
            case MAX7219_OPMODE:
                max7219_set_op_mode (buf[1], buf[2]);
                break;
            case MAX7219_DISPTEST:
                max7219_exec_disp_test (buf[1], buf[2]);
                break;
        }
    }

    return to_copy - not_copied;
}

/**
 * struct file_operations
 */
static struct file_operations fops =
{
    owner   : THIS_MODULE,
    open    : max7219_open,
    release : max7219_close,
    write   : max7219_write
};

/**
 * max7219_init
 * ============
 * <B>max7219_init</B> wird beim Laden des Treibers/Moduls aufgerufen, z.B. bei
 * <B>insmod</B>, <B>modprobe</B>.
 */
static int __init max7219_init (void)
{
    if (alloc_chrdev_region(&max7219_majmin, 0, 1, DEVICE_NAME) < 0)
    {
        pr_err ("%s: alloc_chrdev_region() failed.\n", DEVICE_NAME);
        return -EIO;
    }

    max7219_object = cdev_alloc();
    if (max7219_object == NULL)
    {
        goto free_device_number;
    }

    max7219_object->owner = THIS_MODULE;
    max7219_object->ops   = &fops;

    if (cdev_add (max7219_object, max7219_majmin, 1))
    {
        goto free_cdev;
    }

    max7219_class = class_create(THIS_MODULE, DEVICE_NAME);

    if (IS_ERR(max7219_class))
    {
        pr_err("%s: no udev support\n", DEVICE_NAME);
        goto free_cdev;
    }

    max7219_dev = device_create(max7219_class, NULL, max7219_majmin, NULL,
                                "%s", DEVICE_NAME);
    dev_info (max7219_dev, "max7219_init_called\n");

    return 0;

free_cdev:
    kobject_put(&max7219_object->kobj);

free_device_number:
    unregister_chrdev_region(max7219_majmin, 1);
    return -EIO;
}

/**
 * max7219_exit
 * ============
 * <B>max7219_exit</B> wird aufgerufen, wenn der Treiber oder das Modul entladen
 * werden.
 */
static void __exit max7219_exit (void)
{
    max7219_set_op_mode (0, OPMODE_SHUTDOWN);
    dev_info(max7219_dev, "max7219_exit called\n");
    device_destroy(max7219_class, max7219_majmin);
    class_destroy(max7219_class);
    cdev_del(max7219_object);
    unregister_chrdev_region(max7219_majmin, 1);
    return;
}

/**
 * max7219_clock_out_bit_data
 * ==========================
 * "Taktet" die Bitdaten zum Geraet.
 *
 * Der an den Treiber uebergebene Wert wird als Dualzahl (16 Bit) angezeigt.
 * Zuerst wird das MSB angezeigt, gefolgt von einem Leerzeichen und dann dem
 * LSB.
 */
static void max7219_clock_out_bit_data (uint16_t val)
{
    uint8_t  bitPos = MAX_BITPOS;
    uint8_t  sendBit;
    uint16_t mask = 0;

#ifdef DEBUG
    printk (KERN_ALERT "max7219_clock_out_bit_data = 0x%x\n", val);
#endif // DEBUG

    while (bitPos > 0)
    {
        mask = 1 << (bitPos - 1);
        sendBit = val & mask;    // Maske anwenden.

        // Sende den Wert entsprechend der Maske an GPIO23_DIN
//        gpio_set_value (GPIO23_DIN, ((sendBit) ? HIGH : LOW));
        if (val & mask)
        {
            gpio_set_value (GPIO23_DIN, HIGH);
        }
        else
        {
            gpio_set_value (GPIO23_DIN, LOW);
        }

        // Taktimpuls erzeugen. Laut Datenblatt muessen LOW und HIGH jeweils
        // mindestens 50 ns anstehen --> Hier gewaehlt: 1 µs
        gpio_set_value (GPIO25_CLK, HIGH);
        gpio_set_value (GPIO25_CLK, LOW);

        bitPos -= 1;  // Bitposition fuer das naechste Bit bestimmen.
    }
}


//
//================= Interne Hilfsfunktionen des Treibers =================
//


/**
 * printBin
 * ========
 * Debug-Funktion, die waehrend der Entwicklung verwendet wird.
 *
 * Der an den Treiber uebergebene Wert wird als Dualzahl (16 Bit) angezeigt.
 * Zuerst wird das MSB angezeigt, gefolgt von einem Leerzeichen und dann dem
 * LSB.
 */
static void printBin(uint16_t val)
{
    char     bin[18];
    uint16_t bit;
    uint16_t mask;
    uint16_t index = 0;

    bitPos = 16;
    while (bitPos > 0)
    {
        mask = 1 << (bitPos - 1);
        bit  = val & mask;
        bin [index++] = (bit) ? '1' : '0';

        if (index == 8)
        {
            bin [index++] = ' ';
        }

        --bitPos;   // Bit-Position fuer das naechste Bit ermitteln.
    }
    bin [index] = '\0';

#ifdef DEBUG
    printk (KERN_ALERT "%s\n", bin);
#endif // DEBUG
}

/**
 * max7219_send_noop
 * =================
 * <B>max7219_send_noop</B> wird in einer Kette von kaskadierten MAX7219-Chips
 * immer dann aufgerufen, wenn der in einer der Komfortfunktionen gesetzte Wert
 * <B>pos</B> ungleich 0 ist.
 */
static void max7219_send_noop (void)
{
#ifdef DEBUG
    printBin (0x0000);
#endif // DEBUG
    max7219_clock_out_bit_data (0x0000);
}

/**
 * max7219_make_data
 * =================
 * @brief <B>max7219_make_data</B> wird fuer alle Komfortfunktionen aufgerufen.
 * <B>max7219_make_data</B> wird abhaengig von der Gesamtzahl der verketteten
 * MAX7219-Chips fuer jede der Komfortfunktionen mehrfach aufgerufen.
 *
 * Die Matrizen/Ziffern werden beginnend bei Position 1 gezaehlt und von links
 * nach rechts mit den entsprechenden Daten "versorgt". Der Parameter <B>pos</B>
 * bestimmt die Position eines MAX7219, dessen Wert geaendert werden soll. Hat
 * <B>pos</B> den Wert 0, so wird allen verketteten MAX7219 der gleiche Wert
 * zugewiesen.
 *
 * @param[in]   pos    Position der Matrix/Ziffer der 7-Segment-Anzeige.
 * @param[in]   value  16-Bit-Wert, der an die MAX7219-Chips gesendet wird.
 * @param[out]  -
 */
static void max7219_make_data (uint8_t pos, uint16_t value)
{
    uint8_t count;

#ifdef DEBUG
    printk (KERN_ALERT "max7219_count = %d\n", max7219_count);
    printk (KERN_ALERT "pos           = %d\n", pos);
    printk (KERN_ALERT "value         = 0x%x\n", value);
#endif // DEBUG

    // Ist pos == 0, so wird value an alle verketteten MAX7219-Chips gesendet.
    if (pos == 0)
    {
        for (count = 0; count < max7219_count; count++)
        {
            max7219_clock_out_bit_data (value);
#ifdef DEBUG
            printBin(value);
#endif // DEBUG
        }
    }
    // Ist pos <> 0, so wird nur der Zustand des durch pos angegebenen MAX7219
    // veraendert. Alle anderen MAX7219 bleiben unveraendert.
    else
    {
        // Sendet Daten, beginnend beim letzten MAX7219 (ganz rechts)
        for (count = max7219_count; count > pos; count--)
        {
            max7219_send_noop();
        }

        // pos wurde erreicht --> Sende gewuenschtes Bitmuster
        max7219_clock_out_bit_data (value);
        count--;

        // Sende NOOP an alle MAX7219, die vor pos angeordnet sind.
        while (count > 0)
        {
            max7219_send_noop ();
            count--;
        }
    }
}

//
//================= Zusatzinformationen des Treibers =================
//

/**
 * Zusatzinformationen, teilweise vorgeschrieben.
 */
module_init (max7219_init);
module_exit (max7219_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR ("Ralf Jesse");
MODULE_DESCRIPTION("Kernelmodul zur Ansteuerung eines Schieberegisters vom Typ MAXIM 7219.");
