#ifndef HD44780_H_INCLUDED
#define HD44780_H_INCLUDED

// Allgemeine Definitionen
#define DEVICE_NAME             "hd44780"
#define LOW                            0
#define HIGH                           1
#define OFF                            0
#define ON                             1

// Bitpositionen
#define BIT0                           0
#define BIT1                           1
#define BIT2                           2
#define BIT3                           3
#define BIT4                           4
#define BIT5                           5
#define BIT6                           6
#define BIT7                           7

// Datenbits: Zuordnung
#define BIT_DB4                     0x01
#define BIT_DB5                     0x02
#define BIT_DB6                     0x04
#define BIT_DB7                     0x08

// HD44780: Umschaltung zwischen Kommando- und Textmodus
#define HD44780_MODE_CTRL           0x00
#define HD44780_MODE_TEXT           0x01

// Funktionsnummern
#define CLEAR_DISPLAY               0x01
#define CURSOR_HOME                 0x02
#define DISPLAY_ENABLE              0x03
#define CURSOR_MOVE_DIR             0x04
#define CURSOR_BLINK_MODE           0x05
#define CURSOR_ON_OFF               0x06
#define DISPLAY_ON_OFF              0x07
#define DISPLAY_SHIFT_CURSOR_MOVE   0x08
#define CURSOR_MOVE_LEFT_RIGHT      0x09
#define FONT_SELECT                 0x0A
#define NUM_LINES                   0x0B
#define SET_DATA_WIDTH              0x0C
#define SET_LINE_NUM                0x0D
#define SET_TEXT_ALIGN              0x0E
#define SET_TICKER_MODE             0x0F
#define SET_TICKER_DIR              0x10
#define WRITE_TEXT                  0x11

// Konstante Werte
#define HD44780_DATA_4_BIT          0x03
#define HD44780_DATA_8_BIT          0x30
#define HD44780_CLEAR_DISPLAY       0x01
#define HD44780_CURSOR_HOME         0x02
#define HD44780_SET_CG_RAM_ADDR     0x40
#define HD44780_SET_DD_RAM_ADDR     0x80

// Bits im Funktionsblock DM (Display Mode)
#define HD44780_DM_BLINK_ON         BIT0
#define HD44780_DM_BLINK_OFF        BIT0
#define HD44780_DM_CURSOR_ON        BIT1
#define HD44780_DM_CURSOR_OFF       BIT1
#define HD44780_DM_DISPLAY_ON       BIT2
#define HD44780_DM_DISPLAY_OFF      BIT2

// Bits im Funktionsblock SF (Set Function)
#define HD44780_SF_FONT_SMALL       BIT2
#define HD44780_SF_FONT_BIG         BIT2
#define HD44780_SF_MODE_1_LINE_ON   BIT3
#define HD44780_SF_MODE_2_LINES_ON  BIT3
#define HD44780_SF_MODE_4_BIT_ON    BIT4
#define HD44780_SF_MODE_8_BIT_ON    BIT4

// Fehlercodes
#define HD44780_INIT_FAILED         1000
#define HD44780_INVALID_COMMAND     1001
#define HD44780_INVALID_ARGUMENT    1002
#define HD44780_GPIO_REQ_FAILED     1003
#define HD44780_INVALID_BIT_MODE    1004

// Verwendete GPIO-Ports (Broadcom-Belegung)
#define HD44780_RS                    23    // PIN 16
#define HD44780_ENABLE                25    // PIN 22
#define HD44780_DB4                    5    // PIN 29
#define HD44780_DB5                    6    // PIN 31
#define HD44780_DB6                    7    // PIN 26
#define HD44780_DB7                    8    // PIN 24

// Abhaengig vom gewaehlten LC-Display (1- / 2- / 4-zeilig)
#define LCD_LINE_1                  0x00
#define LCD_LINE_2                  0x01
#define LCD_LINE_3                  0x02
#define LCD_LINE_4                  0x03
#define HD44780_CHARS_PER_LINE        16

// Offsets zum gezielten Schreiben in eine bestimmte Zeile
#define LCD_OFFS_LINE_1             0x80
#define LCD_OFFS_LINE_2             0xC0
#define LCD_OFFS_LINE_3             0x90
#define LCD_OFFS_LINE_4             0xD0
#define SET_LINE_1                  0x8F


// Treiberglobale Variablen
static        char    data_buffer [100];
static        dev_t   hd44780_device_number;
static struct cdev   *hd44780_object;
static struct class  *hd44780_class;
static struct device *hd44780_dev;

/**
 * Die folgenden Variablen speichern die letzte Einstellung des jeweiligen
 * Funktionsblockes. Die Funktionsbloecke unterscheiden sich voneinander
 * durch jeweils ein Bit, das immer den Wert '1' hat. Die einzelnen Funtionen
 * innerhalb eines Funktionsblockes werden durch Bits mit einer kleineren
 * Wertigkeit bestimmt.
 */
static uint8_t hd44780_dm_val  = 0x08; // B3 = 1
static uint8_t hd44780_sf_val  = 0x20; // B5 = 1

// Die Struktur 'struct gpio' ist in linux/gpio.h deklariert!
static struct gpio hd44780_gpios[] =
{
    { HD44780_RS,     GPIOF_DIR_OUT, "RS" },
    { HD44780_ENABLE, GPIOF_DIR_OUT, "Enable" },
    { HD44780_DB4,    GPIOF_DIR_OUT, "Data DB4" },
    { HD44780_DB5,    GPIOF_DIR_OUT, "Data DB5" },
    { HD44780_DB6,    GPIOF_DIR_OUT, "Data DB6" },
    { HD44780_DB7,    GPIOF_DIR_OUT, "Data DB7" },
};

// Funktionsprototypen f. Standardtreiberfunktionen
static int  __init hd44780_init (void);
static void __exit hd44780_exit (void);
static int         hd44780_close (struct inode *, struct file *);
static ssize_t     hd44780_write (struct file *,
                                  const char __user *,
                                  size_t,
                                  loff_t *);

// Prototypen fuer treiberinterne Funktionen
static int  hd44780_lcd_write (int, uint8_t);
static void hd44780_write_nibbles (int, uint8_t );

// Prototypen fuer HD44780-Steuerkommandos
static int hd44780_pre_init (int, uint8_t);
static int hd44780_disp_clear (int);
static int hd44780_show_cursor (int, uint8_t);
static int hd44780_cursor_home (int);
static int hd44780_cursor_blink (int, uint8_t);
static int hd44780_set_line (int, uint8_t);
static int hd44780_ticker_mode (int, uint8_t);
static int hd44780_ticker_dir (int, uint8_t);
static int hd44780_text_alignment (int, uint8_t);
static int hd44780_write_text (int, const char *);


#endif // HD44780_H_INCLUDED
