#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/ctype.h>
#include <asm/uaccess.h>

#include "hd44780.h"

/**
 * hd44780_open
 * ============
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn dort die
 * open()-Funktion verwendet wird. Sie fordert die benoetigten GPIO-Ports an
 * und konfiguriert diese.
 */
static int hd44780_open (struct inode *device_file, struct file *instance)
{
   int err;

    // Anforderung aller GPIOs (Steuer- und Datenleitungen fuer HD44780)
    err = gpio_request_array (hd44780_gpios, ARRAY_SIZE (hd44780_gpios));
    if (err)
    {
        printk (KERN_ALERT "02 Einer oder mehrere der GPIO-Ports konnten \
                nicht zugewiesen werden.\n");
        return -HD44780_GPIO_REQ_FAILED;
    }

    // Alle Aufgaben erfolgreich abgeschlossen.
    return 0;
}

/**
 * hd44780_close
 * =============
 * Diese Funktion wird aus dem Userspace-Programm aufgerufen, wenn das
 * Userspace-Programm beendet wird. Dazu muss das User-Programm die close()-
 * Funktion aufrufen.
 */
static int hd44780_close (struct inode *device_file, struct file *instance)
{
    dev_info (hd44780_dev, "hd44780_close called\n");
    gpio_free_array( hd44780_gpios, ARRAY_SIZE (hd44780_gpios));

    return 0;
}

/**
 * hd44780_write_nibbles
 * =====================
 * Sendet zwei Nibbles (4 Bit) an den LCD-Controller HD44780. Der Wert data_type
 * bestimmt, ob es sich bei den Daten um Steuerungskommandos fuer den HD44780
 * oder um anzuzeigende Daten handelt.
 *
 * @param data_type  HD44780_MODE_CTRL oder HD44780_MODE_TEXT
 * @param lcdData   Daten, die an den Controller gesendet werden
 */
static void hd44780_write_nibbles (int data_type, uint8_t lcdData)
{
    // HD44780: Kommando? --> Setze HD44780_RS = LOW
    // HD44780: Text?     --> Setze HD44780_RS = HIGH
    gpio_set_value (HD44780_RS,
                   (data_type & HD44780_MODE_TEXT) ?
                   HD44780_MODE_TEXT : HD44780_MODE_CTRL);

    udelay (1ul);

    // Ermittlung des Bitmusters fuer die Anschluesse DB4 bis DB7
    // des Controllers aus dem um vier Bit nach links geschobenen
    // Daten.
    gpio_set_value (HD44780_DB4, (lcdData & BIT_DB4) ? HIGH : LOW);
    gpio_set_value (HD44780_DB5, (lcdData & BIT_DB5) ? HIGH : LOW);
    gpio_set_value (HD44780_DB6, (lcdData & BIT_DB6) ? HIGH : LOW);
    gpio_set_value (HD44780_DB7, (lcdData & BIT_DB7) ? HIGH : LOW);

    gpio_set_value (HD44780_ENABLE, HIGH);
    udelay (40);
    gpio_set_value (HD44780_ENABLE, LOW);
}

/**
 * hd44780_lcd_write
 * =================
 * 1. Sendet zuerst das High-Nibble.
 * 2. Senden im Anschluss das Low-Nibble.
 */
static int  hd44780_lcd_write (int data_type, uint8_t value)
{
    hd44780_write_nibbles (data_type, value >> 4);
    udelay (50);
    hd44780_write_nibbles (data_type, value & 0x0F);
    udelay (50);

    return 0;
}

/**
 * hd44780_pre_init
 * ================
 * Pre-Initialisierung der Datenuebertragung. Optionen sind
 * HD44780_PRE_INIT_4_BIT und HD44780_PRE_INIT_8_BIT.
 */
static int hd44780_pre_init (int data_type, uint8_t mode)
{
    // Nach dem Einschalten der Spannungsversorgung mind. 30 ms warten.
    // Hier gewaehlt: 40 ms.
    mdelay (40);

    // 3x Initialisierungssequenz senden.
    hd44780_lcd_write(data_type, mode);
    msleep (5);
    hd44780_lcd_write(data_type, mode);
    udelay (100);;
    hd44780_lcd_write(data_type, mode);
    msleep (5);

    hd44780_lcd_write (data_type, HD44780_CURSOR_HOME);
    msleep (5);

    hd44780_sf_val |= (1 << HD44780_SF_MODE_2_LINES_ON);    // Bit 3 = 1
    hd44780_sf_val &= ~ (1 << HD44780_SF_MODE_4_BIT_ON);    // Bit 4 = 0
    hd44780_lcd_write (data_type, hd44780_sf_val);
    msleep (2);

    hd44780_lcd_write (data_type, HD44780_CLEAR_DISPLAY);
    msleep (2);

    hd44780_dm_val |= (1 << HD44780_DM_DISPLAY_ON) |
                      (1 << HD44780_DM_CURSOR_ON);
    hd44780_dm_val |= (1 << HD44780_DM_BLINK_ON);
    hd44780_lcd_write (data_type, hd44780_dm_val);

    mode = HD44780_SET_DD_RAM_ADDR | 0x80;
    hd44780_lcd_write (data_type, mode);

    return 0;
}

/**
 * hd44780_disp_clear
 * ==================
 * Loescht die Anzeige des gesamten Displays.
 */
static int hd44780_disp_clear (int data_type)
{
    hd44780_lcd_write(data_type, HD44780_CLEAR_DISPLAY);

    return 0;
}

/**
 * hd44780_cursor_home
 * ===================
 * Der naechste Text wird in die erste Spalte der ersten Zeile geschrieben.
 * Wird der Cursor angezeigt (--> Funktion hd44780_show_cursor() ), dann wird
 * dieser ebenfalls in Spalte 1 von Zeile 1 gesetzt.
 */
static int hd44780_cursor_home (int data_type)
{
    hd44780_lcd_write(data_type, HD44780_CURSOR_HOME);

    return 0;
}

/**
 * hd44780_show_cursor
 * ===================
 * Schaltet den Cursor ein oder aus.
 */
static int hd44780_show_cursor (int data_type, uint8_t value)
{
    // Bei einem ungueltigen Argument bleibt die urspruengliche Einstellung
    // erhalten.
    printk (KERN_ALERT "value = %d\n", value);
    if (value == OFF)
    {
        hd44780_dm_val &= ~(1 << HD44780_DM_CURSOR_OFF);
    }
    else if (value == ON)
    {
        hd44780_dm_val |= (1 << HD44780_DM_CURSOR_ON);
    }
    hd44780_lcd_write (data_type, hd44780_dm_val);

    return 0;
}

/**
 * hd44780_cursor_blink
 * ====================
 * Legt fest, ob der Cursor blinkt oder ob er statisch erscheint.
 */
static int hd44780_cursor_blink (int data_type, uint8_t value)
{
    // Bei einem ungueltigen Kommando bleibt die urspruengliche Einstellung
    // erhalten.
    switch (value)
    {
        case OFF:
            printk (KERN_ALERT "\n");
            hd44780_dm_val &= ~(1 << HD44780_DM_BLINK_OFF);
            break;

        case ON:
            printk (KERN_ALERT "\n");
            hd44780_dm_val |= (1 << HD44780_DM_BLINK_ON);
            break;
    }

    hd44780_lcd_write (HD44780_MODE_CTRL, hd44780_dm_val);

    return 0;
}

/**
 * hd44780_cursor_move_dir
 * =======================
 * Legt die Bewegungsrichtung des Cursors fest, falls dieser angezeigt wird.
 */
static int hd44780_cursor_move_dir (int data_type, uint8_t value)
{
    // Not yet implemented

    return 0;
}

/**
 * hd44780_cursor_move_dir
 * =======================
 * Bewegt den Cursor in die mit hd44780_cursor_move_dir festgelegte Richtung.
 */
static int hd44780_cursor_move (int data_type)
{
    // Not yet implemented

    return 0;
}

/**
 * hd44780_set_num_lines
 * =====================
 * Funktionen fuer die Basiskonfiguration des Displays.
 *
 * Die Funktion wird derzeit nicht benoetigt, weil die Anzahl der Zeilen in
 * hd44780.h festgelegt ist.
 */
static int hd44780_set_num_lines (int data_type, uint8_t num_lines)
{
    // Not yest implemented

    return 0;
}

/**
 * hd44780_set_line
 * ================
 * Bestimmt die Zeile, in der der naechste Text angezeigt werden soll.
 */
static int hd44780_set_line (int data_type, uint8_t line_num)
{
    uint8_t line;

    printk (KERN_ALERT "line_num = %d\n", line_num);
    if ((line_num < 1) || (line_num > 4))
    {
        line_num = 1;
    }

    // Hier werden die Bits 6, 5 und 4 auf '0' gesetzt. Hierdurch bleiben
    // zuvor gesetzte Attribute (Bits 0 bis 3) erhalten. Bit 7 hat fuer alle
    // Zeilen immer den Wert '1'. Anschliessend wird die Ausgabezeile fuer den
    // Text durch eine logische Oder-Verknüpfung endgueltig bestimmt.
    hd44780_dm_val &= SET_LINE_1;
    switch (line_num)
    {
        case LCD_LINE_1:
            line = LCD_OFFS_LINE_1;
            break;

        case LCD_LINE_2:
            line = LCD_OFFS_LINE_2;
            break;

        case LCD_LINE_3:
            line = LCD_OFFS_LINE_3;
            break;

        case LCD_LINE_4:
            line = LCD_OFFS_LINE_4;
            break;
    }
    printk (KERN_ALERT "line = %d\n", line);
    hd44780_lcd_write (data_type, line);

    return 0;
}

/**
 * hd44780_display_enable
 * ======================
 * Aktiviert oder deaktiviert das Display.
 */
static int hd44780_display_enable (int data_type, uint8_t value)
{
    // Not yet implemented
    return 0;
}

/**
 * hd44780_display_on_off
 * ======================
 * Schaltet das Display ein oder aus.
 */
static int hd44780_display_on_off (int data_type, uint8_t value)
{
    // Not yet implemented

    return 0;
}

/**
 * hd44780_disp_shift_cursor_move
 * ==============================
 * Wechselt zwischen den Modi "Display Shift" bzw. "Cursor On/Off".
 */
static int hd44780_disp_shift_cursor_move (int data_type, uint8_t mode)
{
    // Not yet implemented

    return 0;
}

/**
 * hd44780_select_font
 * ===================
 * Dient zur Auswahl der unterstuetzten Fonts (5x7 oder 5x10).
 */
static int hd44780_select_font (int data_type, uint8_t font)
{
    // Not yet implemented

    return 0;
}

/**
 * hd44780_text_alignment
 * ======================
 * Stellt die Ausrichtung des Textes (linksbuendig, zentriert oder rechts-
 * buendig ein.
 */
static int hd44780_text_alignment (int data_type, uint8_t align)
{
    // Not yet implemented

    return 0;
}

/**
 * hd44780_ticker_mode
 * ===================
 * Aktiviert/Deaktiviert den Tickermodus. Der Text wandert in Form einer
 * Laufschrift in Richtung der mit hd44780_ticker_dir eingestellten Richtung.
 */
static int hd44780_ticker_mode (int data_type, uint8_t mode)
{
    // Not yet implemented

    return 0;
}

/**
 * hd44780_ticker_dir
 * ==================
 * Legt die Laufrichtung des Textes fest.
 */
static int hd44780_ticker_dir (int data_type, uint8_t mode)
{
    // Not yet implemented

    return 0;
}
/**
 * hd44780_write_text
 * ==================
 * Sendet den angegebenen Text in die aktuell eingestellte Zeile.
 */
static int hd44780_write_text (int data_type, const char *text)
{
    uint8_t index;
    uint8_t len;

    len = strlen (text);
    if (len > HD44780_CHARS_PER_LINE)
    {
        len = HD44780_CHARS_PER_LINE;
    }

    for (index = 0; index < len; index++)
    {
        hd44780_lcd_write (HD44780_MODE_TEXT, text [index]);
    }

    return 0;
}

/**
 * hd44780_write
 * =============
 * Diese Funktion wird aus dem Userspace aufgerufen, z.B. bei
 * echo "text" > /dev/treiber.
 */
static ssize_t hd44780_write (struct file *instance,
                              const char __user *user,
                              size_t count,
                              loff_t *offset)
{
    unsigned long not_copied, to_copy;

    // Fuer den Austausch von Daten zwischen User- und Kernelspace wird ein
    // statischer Buffer vom Typ 'static char' verwendet. Dieser Buffer wird
    // zunaechst geloescht, da noch Werte aus der letzten Operation enthalten
    // sein koennen.
    memset (data_buffer, 0, sizeof (data_buffer));

    to_copy = min (count, sizeof (data_buffer));
    not_copied = copy_from_user (data_buffer, user, to_copy);

    if (iscntrl (data_buffer[0]))
    {
        switch (data_buffer[0])
        {
            case CLEAR_DISPLAY:
                hd44780_disp_clear (HD44780_MODE_CTRL);
                break;

            case CURSOR_HOME:
                hd44780_cursor_home (HD44780_MODE_CTRL);
                break;

            case DISPLAY_ENABLE:
                hd44780_display_enable (HD44780_MODE_CTRL,
                                       (uint8_t) data_buffer [1]);
                break;

            case CURSOR_MOVE_DIR:
                hd44780_cursor_move_dir (HD44780_MODE_CTRL,
                                        (uint8_t) data_buffer [1]);
                break;

            case CURSOR_BLINK_MODE:
                hd44780_cursor_blink (HD44780_MODE_CTRL, data_buffer[1]);
                break;

            case CURSOR_ON_OFF:
                hd44780_show_cursor (HD44780_MODE_CTRL, data_buffer [1]);
                break;

            case DISPLAY_ON_OFF:
                hd44780_display_on_off (HD44780_MODE_CTRL,
                                       (uint8_t) data_buffer [1]);
                break;

            case DISPLAY_SHIFT_CURSOR_MOVE:
                hd44780_disp_shift_cursor_move (HD44780_MODE_CTRL,
                                            (uint8_t) data_buffer [1]);
                break;

            case CURSOR_MOVE_LEFT_RIGHT:
                hd44780_cursor_move (HD44780_MODE_CTRL);
                break;

            case FONT_SELECT:
                hd44780_select_font (HD44780_MODE_CTRL,
                                    (uint8_t) data_buffer [1]);
                break;

            case NUM_LINES:
                hd44780_set_num_lines (HD44780_MODE_CTRL,
                                      (uint8_t) data_buffer [1]);
                break;

            case SET_DATA_WIDTH:
                hd44780_pre_init (HD44780_MODE_CTRL, (uint8_t) data_buffer [1]);
                break;

            case SET_LINE_NUM:
                hd44780_set_line (HD44780_MODE_CTRL, (uint8_t) data_buffer [1]);
                break;

            case SET_TEXT_ALIGN:
                hd44780_text_alignment (HD44780_MODE_CTRL,
                                       (uint8_t) data_buffer [1]);
                break;

            case SET_TICKER_MODE:
                hd44780_ticker_mode (HD44780_MODE_CTRL,
                                    (uint8_t) data_buffer[1]);
                break;

            case SET_TICKER_DIR:
                hd44780_ticker_dir (HD44780_MODE_CTRL,
                                (uint8_t) data_buffer [1]);
                break;

            case WRITE_TEXT:
                hd44780_write_text (HD44780_MODE_TEXT, data_buffer);
                break;

            default:
                printk (KERN_ALERT "Invalid command or argument.\n");
                printk (KERN_ALERT "Function:Argument %d:%d.\n",
                        data_buffer [0], data_buffer [1]);
                memset (data_buffer, 0, sizeof (data_buffer));
                return -HD44780_INVALID_COMMAND;
        }
    }
    else
    {
        printk (KERN_ALERT "else-Zweig: %s\n", data_buffer);
        hd44780_write_text (HD44780_MODE_TEXT, data_buffer);
    }


    return to_copy - not_copied;
}

/**
 * struct file_operations
 */
static struct file_operations fops =
{
    owner   : THIS_MODULE,
    open    : hd44780_open,
    release : hd44780_close,
    write   : hd44780_write
};

/**
 * hd44780_init
 * ============
 * Diese Funktion wird beim Laden des Moduls mit insmod oder modprobe
 * aufgerufen.
 */
static int __init hd44780_init (void)
{
    if (alloc_chrdev_region (&hd44780_device_number, 0, 1, DEVICE_NAME) < 0)
    {
        printk (KERN_ALERT "alloc_chrdev_region failed with MAJOR = %d\n",
                MAJOR(hd44780_device_number));
        return -EIO;
    }

    hd44780_object = cdev_alloc();
    if (hd44780_object == NULL)
    {
        goto free_device_number;
    }

    hd44780_object->owner = THIS_MODULE;
    hd44780_object->ops   = &fops;

    if (cdev_add (hd44780_object, hd44780_device_number, 1))
    {
        goto free_cdev;
    }

    hd44780_class = class_create(THIS_MODULE, DEVICE_NAME);

    if (IS_ERR(hd44780_class))
    {
        pr_err("%s: no udev support\n", DEVICE_NAME);
        goto free_cdev;
    }

    hd44780_dev = device_create(hd44780_class,
                                NULL,
                                hd44780_device_number,
                                NULL,
                                "%s",
                                DEVICE_NAME);
    dev_info (hd44780_dev, "hd44780_init called\n");


    return 0;

free_cdev:
    kobject_put(&hd44780_object->kobj);

free_device_number:
    unregister_chrdev_region(hd44780_device_number, 1);
    return -EIO;

    // Alle Aufgaben erfolgreich abgeschlossen.
    return 0;
}

/**
 * hd44780_exit
 * ============
 * Wird aufgerufen, wenn das Modul mit rmmod entladen wird.
 */
static void __exit hd44780_exit (void)
{
    gpio_free_array( hd44780_gpios, ARRAY_SIZE (hd44780_gpios));
    dev_info(hd44780_dev, "hd44780_exit called\n");
    device_destroy(hd44780_class, hd44780_device_number);
    class_destroy(hd44780_class);
    cdev_del(hd44780_object);
    unregister_chrdev_region(hd44780_device_number, 1);

    return;
}


/**
 * Zusatzinformationen, teilweise zwingend erforderlich.
 */
module_init (hd44780_init);
module_exit (hd44780_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR ("R. Jesse");
MODULE_DESCRIPTION("Kernelmodul zur Ansteuerung eines LCD-Controllers vom \
                    Typ HD-44780.");
