/**
 * @brief Sucht die I2C-Adressen externer I2C-Komponenten.
 *
 * Das Beispiel existiert ausschliesslich als MCAL-Version.
 *
 * @par
 * Verwendete Anschlusspins:
 *     PB8 : I2C1 SCL
 *     PB9 : I2C1 SDA
 *
 * @note
 * !!!WICHTIG!!!
 * Verwenden Sie unbedingt externe Pull-up-Widerstaende (PB8/PB9 muessen dann
 * zwingend im Open-Drain-Modus konfiguriert sein). Mit den internen Pull-up-
 * Widerstaenden funktioniert I2C beim STM32 nicht zuverlaessig!
 */

#include <stm32f4xx.h>
#include <stdint.h>
#include <stdbool.h>

#include <mcalGPIO.h>
#include <mcalI2C.h>
#include <mcalSystem.h>
#include <mcalSysTick.h>

bool timerTrigger = false;

int main(void)
{
    GPIO_TypeDef *port = GPIOB;
    I2C_TypeDef  *i2c  = I2C1;
    uint8_t       devAddr[128] = { 0 };
    uint8_t       scanAddr = 0x00;
    uint8_t       foundAddr = 0;
    uint8_t       index = 0;
    uint32_t      i2cScanTimer;
    uint32_t      pclock;

    pclock = sysGetPclk1Freq();

    systickInit(SYSTICK_1MS);
    systickSetTicktime(&i2cScanTimer, 10);


    // GPIOB-Bustakt aktivieren wegen der Verwendung von PB8/PB9.
    gpioInitPort(port);
    gpioSelectPinMode(port, PIN8, ALTFUNC);
    gpioSelectAltFunc(port, PIN8, AF4);         // PB8 : I2C1 SCL
    gpioSelectPinMode(port, PIN9, ALTFUNC);
    gpioSelectAltFunc(port, PIN9, AF4);         // PB9 : I2C1 SDA
    gpioSetOutputType(port, PIN8, OPENDRAIN);   // Externe Pull-up-Widerstaende
    gpioSetOutputType(port, PIN9, OPENDRAIN);   // verwenden!!!

    /**
     * Die beiden folgenden Zeilen muessen auskommentiert werden,
     * wenn Sie externe Pull-Up-Widerstände verwenden.
     */
//    gpioSelectPushPullMode(port, PIN8, PULLUP);
//    gpioSelectPushPullMode(port, PIN9, PULLUP);

    // Initialisierung des I2C-Controllers
    i2cSelectI2C(i2c);                          // I2C1: Bustakt aktivieren
    i2cDisableDevice(i2c);
    i2cSetPeripheralClockFreq(i2c, pclock);     // I2C1: Periph. Clk in MHz
    i2cSetDutyCycle(i2c, I2C_DUTY_CYCLE_2);     // I2C1: Duty-cycle einstellen
    i2cSetRiseTime(i2c, 17);                    // I2C1: 17 ist ein bewaehrter Wert
    i2c->CCR |= 0x50;                           // I2C1: Keine MCAL-Funktion
    i2cEnableDevice(i2c);                       // I2C1: Aktivieren

    /* Hauptprogramm: Endlosschleife */
    while(1)
    {
        if (true == timerTrigger)
        {
            systickUpdateTimer(&i2cScanTimer);
        }

        if (isSystickExpired(i2cScanTimer))
        {
            timerTrigger = false;
            foundAddr = i2cFindSlaveAddr(i2c, scanAddr);

            if (foundAddr != 0)
            {
                devAddr[index++] = foundAddr;
            }
            systickSetTicktime(&i2cScanTimer, 10);
            scanAddr++;
            if (scanAddr > 0x7F)
            {
                break;
            }
        }
    }
    return 0;
}
